// import-all.cjs
const admin = require('firebase-admin');

// Point to emulator
process.env.FIRESTORE_EMULATOR_HOST = 'localhost:8080';

// Initialize only if not already done
if (admin.apps.length === 0) {
  admin.initializeApp({
    projectId: 'electronic-log-book-2d67f', // ← Use this!
  });
}

const db = admin.firestore();

// ✅ Load the ACTUAL backup JSON file
const data = require('./firestore-backup.json');

function convertTimestamp(obj) {
  if (obj && typeof obj === 'object' && '_seconds' in obj) {
    return admin.firestore.Timestamp.fromMillis(
      obj._seconds * 1000 + Math.floor((obj._nanoseconds || 0) / 1000000)
    );
  }
  return obj;
}

function cleanDocument(doc) {
  const cleaned = { ...doc };
  delete cleaned.id; // Remove 'id' since we use it as doc ID

  // Recursively convert timestamps
  for (const key in cleaned) {
    if (cleaned[key] && typeof cleaned[key] === 'object') {
      if ('_seconds' in cleaned[key]) {
        cleaned[key] = convertTimestamp(cleaned[key]);
      }
      // Add more nested handling if needed (e.g., arrays, sub-objects)
    }
  }

  return cleaned;
}

async function importCollection(collectionName, docs) {
  if (!Array.isArray(docs)) {
    console.warn(`⚠️ Skipping ${collectionName}: not an array`);
    return;
  }

  console.log(`📦 Importing ${docs.length} documents into "${collectionName}"...`);

  const batchSize = 500; // Firestore batch limit
  for (let i = 0; i < docs.length; i += batchSize) {
    const batch = db.batch();
    const chunk = docs.slice(i, i + batchSize);

    for (const doc of chunk) {
      const docId = doc.id || db.collection(collectionName).doc().id;
      const docRef = db.collection(collectionName).doc(docId);
      const cleanedDoc = cleanDocument(doc);
      batch.set(docRef, cleanedDoc);
    }

    await batch.commit();
    console.log(`  ✅ Batch ${Math.floor(i / batchSize) + 1} committed`);
  }
}

async function main() {
  console.log('🚀 Starting import to Firestore Emulator...\n');

  // Import each top-level key as a collection
  for (const [collectionName, docs] of Object.entries(data)) {
    if (Array.isArray(docs)) {
      await importCollection(collectionName, docs);
    } else {
      console.warn(`⚠️ Skipping "${collectionName}": not an array`);
    }
  }

  console.log('\n🎉 All data imported successfully!');
  process.exit(0);
}

main().catch(err => {
  console.error('❌ Import failed:', err);
  process.exit(1);
});