import { AuthData, RateLimits, Request, RetryConfig } from '../../common/providers/tasks';
import * as options from '../options';
import { Expression } from '../params';
import { HttpsFunction } from './https';
export { AuthData, Request };
export interface TaskQueueOptions extends options.EventHandlerOptions {
    /** How a task should be retried in the event of a non-2xx return. */
    retryConfig?: RetryConfig;
    /** How congestion control should be applied to the function. */
    rateLimits?: RateLimits;
    /**
     * Who can enqueue tasks for this function.
     * If left unspecified, only service accounts which have
     * `roles/cloudtasks.enqueuer` and `roles/cloudfunctions.invoker`
     * will have permissions.
     */
    invoker?: 'private' | string | string[];
    /**
     * Region where functions should be deployed.
     */
    region?: options.SupportedRegion | string;
    /**
     * Amount of memory to allocate to a function.
     * A value of null restores the defaults of 256MB.
     */
    memory?: options.MemoryOption | Expression<number> | null;
    /**
     * Timeout for the function in sections, possible values are 0 to 540.
     * HTTPS functions can specify a higher timeout.
     * A value of null restores the default of 60s
     * The minimum timeout for a gen 2 function is 1s. The maximum timeout for a
     * function depends on the type of function: Event handling functions have a
     * maximum timeout of 540s (9 minutes). HTTPS and callable functions have a
     * maximum timeout of 36,00s (1 hour). Task queue functions have a maximum
     * timeout of 1,800s (30 minutes)
     */
    timeoutSeconds?: number | Expression<number> | null;
    /**
     * Min number of actual instances to be running at a given time.
     * Instances will be billed for memory allocation and 10% of CPU allocation
     * while idle.
     * A value of null restores the default min instances.
     */
    minInstances?: number | Expression<number> | null;
    /**
     * Max number of instances to be running in parallel.
     * A value of null restores the default max instances.
     */
    maxInstances?: number | Expression<number> | null;
    /**
     * Number of requests a function can serve at once.
     * Can only be applied to functions running on Cloud Functions v2.
     * A value of null restores the default concurrency (80 when CPU >= 1, 1 otherwise).
     * Concurrency cannot be set to any value other than 1 if `cpu` is less than 1.
     * The maximum value for concurrency is 1,000.
     */
    concurrency?: number | Expression<number> | null;
    /**
     * Fractional number of CPUs to allocate to a function.
     * Defaults to 1 for functions with <= 2GB RAM and increases for larger memory sizes.
     * This is different from the defaults when using the gcloud utility and is different from
     * the fixed amount assigned in Google Cloud Functions generation 1.
     * To revert to the CPU amounts used in gcloud or in Cloud Functions generation 1, set this
     * to the value "gcf_gen1"
     */
    cpu?: number | 'gcf_gen1';
    /**
     * Connect cloud function to specified VPC connector.
     * A value of null removes the VPC connector
     */
    vpcConnector?: string | null;
    /**
     * Egress settings for VPC connector.
     * A value of null turns off VPC connector egress settings
     */
    vpcConnectorEgressSettings?: options.VpcEgressSetting | null;
    /**
     * Specific service account for the function to run as.
     * A value of null restores the default service account.
     */
    serviceAccount?: string | null;
    /**
     * Ingress settings which control where this function can be called from.
     * A value of null turns off ingress settings.
     */
    ingressSettings?: options.IngressSetting | null;
    /**
     * User labels to set on the function.
     */
    labels?: Record<string, string>;
    secrets?: string[];
    /** Whether failed executions should be delivered again. */
    retry?: boolean;
}
/**
 * A handler for tasks.
 * @typeParam T - The task data interface. Task data is unmarshaled from JSON.
 */
export interface TaskQueueFunction<T = any> extends HttpsFunction {
    /**
     * The callback passed to the `TaskQueueFunction` constructor.
     * @param request - A TaskRequest containing data and auth information.
     * @returns Any return value. Google Cloud Functions will await any promise
     *          before shutting down your function. Resolved return values
     *          are only used for unit testing purposes.
     */
    run(request: Request<T>): void | Promise<void>;
}
/**
 * Creates a handler for tasks sent to a Google Cloud Tasks queue.
 * @param handler - A callback to handle task requests.
 * @typeParam Args - The interface for the request's `data` field.
 * @returns A Cloud Function you can export and deploy.
 */
export declare function onTaskDispatched<Args = any>(handler: (request: Request<Args>) => void | Promise<void>): TaskQueueFunction<Args>;
/**
 * Creates a handler for tasks sent to a Google Cloud Tasks queue.
 * @param options - Configuration for the task queue or Cloud Function.
 * @param handler - A callback to handle task requests.
 * @typeParam Args - The interface for the request's `data` field.
 * @returns A Cloud Function you can export and deploy.
 */
export declare function onTaskDispatched<Args = any>(options: TaskQueueOptions, handler: (request: Request<Args>) => void | Promise<void>): TaskQueueFunction<Args>;
