import asyncHandler from "../../helpers/asyncHandler";
import { Request, Response } from "express";
import { BadRequestError } from "../../utils/ApiError";
import { db, admin } from "../../../firebase";
import {
  S3Client,
  PutObjectCommand,
  GetObjectCommand,
} from "@aws-sdk/client-s3";
import { getSignedUrl } from "@aws-sdk/s3-request-presigner";

// Make S3 optional; if env is missing, skip S3 operations gracefully
const AWS_REGION = process.env.AWS_REGION;
const AWS_ACCESS_KEY_ID = process.env.AWS_ACCESS_KEY_ID;
const AWS_SECRET_ACCESS_KEY = process.env.AWS_SECRET_ACCESS_KEY;
const AWS_BUCKET_NAME = process.env.AWS_BUCKET_NAME;
const hasS3 = Boolean(
  AWS_REGION && AWS_ACCESS_KEY_ID && AWS_SECRET_ACCESS_KEY && AWS_BUCKET_NAME
);

const s3 = hasS3
  ? new S3Client({
      region: AWS_REGION,
      credentials: {
        accessKeyId: AWS_ACCESS_KEY_ID as string,
        secretAccessKey: AWS_SECRET_ACCESS_KEY as string,
      },
    })
  : null;

export default class DamageController {
  
  // Helper to normalize imageSrc: convert web paths (/images/charts/1.png) to chart IDs (chart_1)
  private normalizeImageSrc(imageSrc: string | undefined | null): string | null {
    if (!imageSrc) return null;
    // If already in chart_ format, return as-is
    if (imageSrc.startsWith('chart_')) return imageSrc;
    // Extract number from path like /images/charts/1.png or images/charts/1.png
    const match = imageSrc.match(/(\d+)\.png$/);
    if (match) {
      return `chart_${match[1]}`;
    }
    // Return original if can't normalize
    return imageSrc;
  }

  create = asyncHandler(async (req: any, res: Response) => {
    const { name, status, location, description, marker, imageSrc } = req.body;
    const files = req.files;
    
    // Normalize imageSrc to consistent format (chart_1, chart_2, etc.)
    const normalizedImageSrc = this.normalizeImageSrc(imageSrc);

    let seq = 1;
    try {
      const seqSnap = await db.collection("damages").orderBy("seq", "desc").limit(1).get();
      if (!seqSnap.empty) {
        const lastSeq = seqSnap.docs[0].data().seq;
        seq = (typeof lastSeq === 'number' ? lastSeq : 0) + 1;
      }
    } catch (error: any) {
      // If seq field doesn't exist or ordering fails, use count as fallback
      const countSnap = await db.collection("damages").get();
      seq = countSnap.size + 1;
    }
    const id = `DMG-${seq.toString().padStart(5, "0")}`;

    const fileKeys: string[] = [];
    const fileNames: string[] = [];

    if (files && files.length > 0) {
      if (!hasS3 || !s3) {
        throw new BadRequestError(
          "S3 is not configured. Please set AWS_REGION, AWS_ACCESS_KEY_ID, AWS_SECRET_ACCESS_KEY, AWS_BUCKET_NAME"
        );
      }

      for (const file of files) {
        const safeName = file.originalname.replace(/\s+/g, "_");
        const uniqueName = `${Date.now()}-${Math.random().toString(36).substr(2, 9)}-${safeName}`;

        await s3.send(
          new PutObjectCommand({
            Bucket: AWS_BUCKET_NAME!,
            Key: uniqueName,
            Body: file.buffer,
            ContentType: file.mimetype,
          })
        );

        fileKeys.push(uniqueName);
        fileNames.push(file.originalname);
      }
    }

    const docRef = db.collection("damages").doc();
    await docRef.set({
      id,
      seq,
      name: name || null,
      status: status || "Active",
      date: new Date().toISOString().split("T")[0].split("-").reverse().join("/"),
      location: location || null,
      description: description || null,
      fileKeys: fileKeys.length > 0 ? fileKeys : null,
      fileNames: fileNames.length > 0 ? fileNames : null,
      marker: marker ? (typeof marker === 'string' ? JSON.parse(marker) : marker) : null,
      imageSrc: normalizedImageSrc,
      // Also store original for backward compatibility
      originalImageSrc: imageSrc,
      createdAt: admin.firestore.FieldValue.serverTimestamp(),
    });

    // Return direct object response
    return res.status(200).json({
      success: true,
      message: "Damage saved ✅",
      data: { damageId: docRef.id }
    });
  });

  getAll = asyncHandler(async (req: Request, res: Response) => {
    let snap;
    try {
      snap = await db
        .collection("damages")
        .orderBy("createdAt", "desc")
        .get();
    } catch (error: any) {
      // If orderBy fails (e.g., missing createdAt), fallback to simple get
      snap = await db.collection("damages").get();
    }

    const list = await Promise.all(
      snap.docs.map(async (d) => {
        const data = d.data();
        let fileUrls: string[] = [];

        if (data.fileKeys && Array.isArray(data.fileKeys) && hasS3 && s3) {
          fileUrls = await Promise.all(
            data.fileKeys.map((key: string) =>
              getSignedUrl(
                s3,
                new GetObjectCommand({
                  Bucket: AWS_BUCKET_NAME!,
                  Key: key,
                }),
                { expiresIn: 3600 }
              )
            )
          );
        }

        return {
          firebaseId: d.id,
          ...data,
          fileUrls,
          fileNames: data.fileNames || [],
          // Ensure imageSrc is normalized (prefer normalized, fallback to original)
          imageSrc: data.imageSrc || data.originalImageSrc || null,
        };
      })
    );

    // Return direct array response
    return res.status(200).json(list);
  });

  getOne = asyncHandler(async (req: Request, res: Response) => {
    const { id } = req.params;

    const docRef = db.collection("damages").doc(id);
    const doc = await docRef.get();

    if (!doc.exists) {
      throw new BadRequestError('Damage not found');
    }

    const data = doc.data()!;
    let fileUrls: string[] = [];

    if (data.fileKeys && Array.isArray(data.fileKeys)) {
      fileUrls = await Promise.all(
        data.fileKeys.map((key: string) =>
          getSignedUrl(
            s3,
            new GetObjectCommand({
              Bucket: process.env.AWS_BUCKET_NAME!,
              Key: key,
            }),
            { expiresIn: 3600 }
          )
        )
      );
    }

    // Return direct object response
    return res.status(200).json({
      firebaseId: doc.id,
      ...data,
      fileUrls,
      fileNames: data.fileNames || [],
    });
  });

  update = asyncHandler(async (req: Request, res: Response) => {
    const { id } = req.params;
    const { name, status, location, description } = req.body;

    const docRef = db.collection("damages").doc(id);
    const doc = await docRef.get();

    if (!doc.exists) {
      throw new BadRequestError('Damage not found');
    }

    await docRef.update({
      name: name || null,
      status: status || "Active",
      location: location || null,
      description: description || null,
      updatedAt: admin.firestore.FieldValue.serverTimestamp(),
    });

    // Return direct object response
    return res.status(200).json({
      success: true,
      message: "Damage updated successfully ✅",
      data: { damageId: id }
    });
  });

  delete = asyncHandler(async (req: Request, res: Response) => {
    const { id } = req.params;

    const docRef = db.collection("damages").doc(id);
    const doc = await docRef.get();

    if (!doc.exists) {
      throw new BadRequestError('Damage not found');
    }

    await docRef.delete();

    // Return direct object response
    return res.status(200).json({
      success: true,
      message: "Damage deleted successfully ✅"
    });
  });

  // Optional: Get damages by status
  getByStatus = asyncHandler(async (req: Request, res: Response) => {
    const { status } = req.params;

    let snap;
    try {
      snap = await db
        .collection("damages")
        .where("status", "==", status)
        .orderBy("createdAt", "desc")
        .get();
    } catch (error: any) {
      // If orderBy fails (e.g., missing createdAt), fallback to simple where
      snap = await db.collection("damages").where("status", "==", status).get();
    }

    const list = await Promise.all(
      snap.docs.map(async (d) => {
        const data = d.data();
        let fileUrls: string[] = [];

        if (data.fileKeys && Array.isArray(data.fileKeys) && hasS3 && s3) {
          fileUrls = await Promise.all(
            data.fileKeys.map((key: string) =>
              getSignedUrl(
                s3,
                new GetObjectCommand({
                  Bucket: AWS_BUCKET_NAME!,
                  Key: key,
                }),
                { expiresIn: 3600 }
              )
            )
          );
        }

        return {
          firebaseId: d.id,
          ...data,
          fileUrls,
          fileNames: data.fileNames || [],
          // Ensure imageSrc is normalized (prefer normalized, fallback to original)
          imageSrc: data.imageSrc || data.originalImageSrc || null,
        };
      })
    );

    // Return direct array response
    return res.status(200).json(list);
  });
}