// src/controller/flightController.ts
import asyncHandler from "../../helpers/asyncHandler";
import { Request, Response } from "express";
import { BadRequestError } from "../../utils/ApiError";
import { db, admin } from "../../../firebase";
import { generateNextLogPageNo } from "../../utils/flightUtils";

export default class FlightController {
  /* --------------------------------------------------------------
   *  PRIVATE HELPERS
   * -------------------------------------------------------------- */
  private async recomputeFlightLegs(currentFlightId: string) {
    const batch = db.batch(); 

    // 1. Load every flight with its createdAt timestamp
    const flightsSnap = await db.collection("flights").get();
    const flightData = flightsSnap.docs.map((doc) => ({
      id: doc.id,
      createdAt: doc.data().createdAt,
      isCurrent: doc.data().currentFlight === true,
    }));

    if (flightData.length === 0) return;

    // 2. Sort chronologically
    flightData.sort(
      (a, b) => a.createdAt.toMillis() - b.createdAt.toMillis()
    );

    // 3. Determine base index (current flight)
    let baseIndex = flightData.findIndex(
      (f) => f.id === currentFlightId && f.isCurrent
    );
    if (baseIndex === -1)
      baseIndex = flightData.findIndex((f) => f.isCurrent);
    if (baseIndex === -1) baseIndex = flightData.length - 1; // fallback to newest

    // 4. Load every log and recompute its flightLeg
    const logsSnap = await db.collection("logs").get();
    logsSnap.forEach((logDoc) => {
      const logData = logDoc.data();
      const flight = flightData.find((f) => f.id === logData.flightId);
      if (flight) {
        const flightIndex = flightData.indexOf(flight);
        const flightLeg = flightIndex - baseIndex;

        batch.update(logDoc.ref, {
          flightLeg,
          updatedAt: admin.firestore.FieldValue.serverTimestamp(),
        });
      }
    });

    await batch.commit();
  }

  /* --------------------------------------------------------------
   *  FLIGHT CRUD
   * -------------------------------------------------------------- */
  create = asyncHandler(async (req: Request, res: Response) => {
    const {
      regn,
      fltNo,
      typeOfFlight,
      from,
      to,
      landings,
      offBlock,
      onBlock,
      prevHrs,
      takeOff,
      landing,
      flightTime,
      takeOffDate,
      landingDate,
      totalHrs,
      employeeNo,
      captName,
      acftRelease,
    } = req.body;

    const batch = db.batch();
    const flightRef = db.collection("flights").doc();
    const flightId = flightRef.id;

    // ---- Determine if this is the first flight ever ----
    const existingFlightsSnap = await db.collection("flights").get();
    const isFirstFlight = existingFlightsSnap.empty;

    // ---- Should this flight be the current one? ----
    let makeCurrent = isFirstFlight;
    if (!isFirstFlight) {
      const currentSnap = await db
        .collection("flights")
        .where("currentFlight", "==", true)
        .limit(1)
        .get();
      makeCurrent = currentSnap.empty;
    }

    // ---- Write the flight document ----
    batch.set(flightRef, {
      regn,
      fltNo,
      typeOfFlight,
      from,
      to,
      landings,
      offBlock,
      onBlock,
      prevHrs,
      takeOff,
      landing,
      flightTime,
      takeOffDate,
      landingDate,
      totalHrs,
      employeeNo,
      captName,
      acftRelease: !!acftRelease,
      currentFlight: makeCurrent,
      createdAt: admin.firestore.FieldValue.serverTimestamp(),
      updatedAt: admin.firestore.FieldValue.serverTimestamp(),
    });

    // ---- Create the associated log (page number) ----
    const logsSnap = await db
      .collection("logs")
      .orderBy("createdAt", "desc")
      .get();
    const existingPageNos = logsSnap.docs.map((d) => d.data().logPageNo);
    const logPageNo = generateNextLogPageNo(existingPageNos);

    const logRef = db.collection("logs").doc();
    batch.set(logRef, {
      logPageNo,
      flightId,
      flightLeg: 0, // temporary – will be fixed by recompute
      status: 1,
      createdBy: employeeNo || captName || "Unknown",
      createdAt: admin.firestore.FieldValue.serverTimestamp(),
      updatedAt: admin.firestore.FieldValue.serverTimestamp(),
    });

    await batch.commit();

    // ---- Re-calculate all flightLeg values ----
    await this.recomputeFlightLegs(flightId);

    return res.status(200).json({
      success: true,
      message: "Flight and log saved",
      data: { flightId, logId: logRef.id, logPageNo },
    });
  });

  getAll = asyncHandler(async (_req: Request, res: Response) => {
    const flightsSnap = await db
      .collection("flights")
      .orderBy("createdAt", "desc")
      .get();

    const logsSnap = await db.collection("logs").get();
    const legMap = new Map<string, number>();
    logsSnap.forEach((d) => legMap.set(d.data().flightId, d.data().flightLeg ?? 0));

    const flights = flightsSnap.docs.map((doc) => ({
      id: doc.id,
      ...doc.data(),
      flightLeg: legMap.get(doc.id) ?? 0,
    }));

    return res.status(200).json(flights);
  });

  getOne = asyncHandler(async (req: Request, res: Response) => {
    const { id } = req.params;
    const doc = await db.collection("flights").doc(id).get();
    if (!doc.exists) throw new BadRequestError("Flight not found");

    return res.status(200).json({ id: doc.id, ...doc.data() });
  });

  update = asyncHandler(async (req: Request, res: Response) => {
    const { id } = req.params;
    const flightRef = db.collection("flights").doc(id);
    const snap = await flightRef.get();
    if (!snap.exists) throw new BadRequestError("Flight not found");

    const wasCurrent = snap.data()?.currentFlight === true;

    await flightRef.update({
      ...req.body,
      acftRelease: !!req.body.acftRelease,
      currentFlight: !!req.body.currentFlight,
      updatedAt: admin.firestore.FieldValue.serverTimestamp(),
    });

    const isNowCurrent = !!req.body.currentFlight;
    if (isNowCurrent && !wasCurrent) {
      await this.recomputeFlightLegs(id);
    }

    return res.status(200).json({ success: true, message: "Flight updated" });
  });

  delete = asyncHandler(async (req: Request, res: Response) => {
    const { id } = req.params;
    const flightRef = db.collection("flights").doc(id);
    const doc = await flightRef.get();
    if (!doc.exists) throw new BadRequestError("Flight not found");

    const logSnap = await db
      .collection("logs")
      .where("flightId", "==", id)
      .get();

    const batch = db.batch();
    logSnap.forEach((d) => batch.delete(d.ref));
    batch.delete(flightRef);
    await batch.commit();

    return res.status(200).json({ success: true, message: "Flight deleted" });
  });

  makeCurrent = asyncHandler(async (req: Request, res: Response) => {
    const { id } = req.params;
    const flightRef = db.collection("flights").doc(id);
    const snap = await flightRef.get();
    if (!snap.exists) throw new BadRequestError("Flight not found");

    const batch = db.batch();

    // unset any existing current flight
    const currentSnap = await db
      .collection("flights")
      .where("currentFlight", "==", true)
      .get();
    currentSnap.forEach((d) =>
      batch.update(d.ref, {
        currentFlight: false,
        updatedAt: admin.firestore.FieldValue.serverTimestamp(),
      })
    );

    // set the new current flight
    batch.update(flightRef, {
      currentFlight: true,
      updatedAt: admin.firestore.FieldValue.serverTimestamp(),
    });

    await batch.commit();
    await this.recomputeFlightLegs(id);

    return res.status(200).json({ success: true, message: "Flight set as current" });
  });

  /* --------------------------------------------------------------
   *  DYNAMIC LISTS (Airports + derived REGN / FLT-NO)
   * -------------------------------------------------------------- */
  /** ADD AIRPORT */
  addAirport = asyncHandler(async (req: Request, res: Response) => {
    const { code, name } = req.body;
    if (!code || !name) throw new BadRequestError("Code and name required");

    const ref = db.collection("airports").doc();
    await ref.set({
      code,
      name,
      createdAt: admin.firestore.FieldValue.serverTimestamp(),
    });

    return res.status(200).json({
      success: true,
      message: "Airport added",
      data: { id: ref.id },
    });
  });

  /** GET ALL AIRPORTS */
  getAirports = asyncHandler(async (_req: Request, res: Response) => {
    const snap = await db.collection("airports").get();
    const airports = snap.docs.map((d) => ({ id: d.id, ...d.data() }));
    return res.status(200).json(airports);
  });

  /** DELETE AIRPORT */
  deleteAirport = asyncHandler(async (req: Request, res: Response) => {
    const { id } = req.params;
    const docRef = db.collection("airports").doc(id);
    const doc = await docRef.get();

    if (!doc.exists) throw new BadRequestError("Airport not found");

    await docRef.delete();

    return res.status(200).json({
      success: true,
      message: "Airport deleted successfully",
    });
  });

  /** GET UNIQUE REGNs (derived from flights) */
  getRegns = asyncHandler(async (_req: Request, res: Response) => {
    const snap = await db.collection("flights").orderBy("regn").get();
    const set = new Set<string>();
    snap.forEach((d) => d.data().regn && set.add(d.data().regn));
    return res.status(200).json(Array.from(set).map((regnNo) => ({ regnNo })));
  });

  /** GET UNIQUE FLIGHT NUMBERS (derived from flights) */
  getFlightNos = asyncHandler(async (_req: Request, res: Response) => {
    const snap = await db.collection("flights").orderBy("fltNo").get();
    const set = new Set<string>();
    snap.forEach((d) => d.data().fltNo && set.add(d.data().fltNo));
    return res.status(200).json(Array.from(set).map((flightNo) => ({ flightNo })));
  });
}