import asyncHandler from "../../helpers/asyncHandler";
import { Request, Response } from "express";
import { BadRequestError } from "../../utils/ApiError";
import { db, admin } from "../../../firebase"; // Removed .ts extension
import { normalizeDdType, normalizeMelCat, generateLogPageNo } from "../../utils/logUtils";
import { DeferralService } from "../../services/deferralService";

export default class LogController { 
  
  addOrUpdateLogItem = asyncHandler(async (req: Request, res: Response) => {
    console.log("item req", req.body)
    const { logId } = req.params;
    const { logItem, createdBy } = req.body;

    if (!logItem || !createdBy) {
      throw new BadRequestError("logItem and createdBy are required");
    }

    const logRef = db.collection("logs").doc(logId);
    const logDoc = await logRef.get();
    if (!logDoc.exists) {
      throw new BadRequestError("Log not found");
    }

    const logPageNo = logDoc.data()!.logPageNo;

    let logItemId;
    let isNew = false;
    let displayNumber;

    if (logItem.id) {
      const logItemRef = db.collection("logItems").doc(logItem.id);
      const logItemDoc = await logItemRef.get();
      if (!logItemDoc.exists) {
        throw new BadRequestError("Log item not found");
      }
      logItemId = logItem.id;
      displayNumber = logItemDoc.data()!.displayNumber;
      
      await logItemRef.update({
        raisedBy: logItem.raisedBy || "",
        class: logItem.class || "",
        defectDetails: logItem.defectDetails || "",
        actionDetails: logItem.actionDetails || "",
        ddChecked: logItem.ddChecked || false,
        ddAction: logItem.ddAction || "",
        ddType: normalizeDdType(logItem.ddType) || "",
        ddNo: logItem.ddNo || "",
        melCdlRef: logItem.melCdlRef || "",
        cat: normalizeMelCat(logItem.cat) || "A",
        indInspChecked: logItem.indInspChecked || false,
        sdr: logItem.sdr || false,
        mmsgFc: logItem.mmsgFc || "",
        ata: logItem.ata || "",
        shortSignAuthId: logItem.shortSignAuthId || "",
        shortSignAuthName: logItem.shortSignAuthName || "",
        actionAuthId: logItem.actionAuthId || "",
        actionAuthName: logItem.actionAuthName || "",
        updatedAt: admin.firestore.FieldValue.serverTimestamp(),
      });
    } else {
      const existingItemsSnap = await db.collection("logItems").where("logId", "==", logId).get();
      displayNumber = existingItemsSnap.empty
        ? 1
        : Math.max(...existingItemsSnap.docs.map((doc) => doc.data().displayNumber || 0)) + 1;

      const logItemRef = db.collection("logItems").doc();
      logItemId = logItemRef.id;
      isNew = true;

      await logItemRef.set({
        logId,
        displayNumber,
        raisedBy: logItem.raisedBy || "",
        class: logItem.class || "",
        defectDetails: logItem.defectDetails || "",
        actionDetails: logItem.actionDetails || "",
        ddChecked: logItem.ddChecked || false,
        ddAction: logItem.ddAction || "",
        ddType: normalizeDdType(logItem.ddType) || "",
        ddNo: logItem.ddNo || "",
        melCdlRef: logItem.melCdlRef || "",
        cat: normalizeMelCat(logItem.cat) || "A",
        indInspChecked: logItem.indInspChecked || false,
        sdr: logItem.sdr || false,
        mmsgFc: logItem.mmsgFc || "",
        ata: logItem.ata || "",
        shortSignAuthId: logItem.shortSignAuthId || "",
        shortSignAuthName: logItem.shortSignAuthName || "",
        actionAuthId: logItem.actionAuthId || "",
        actionAuthName: logItem.actionAuthName || "",
        createdAt: admin.firestore.FieldValue.serverTimestamp(),
        updatedAt: admin.firestore.FieldValue.serverTimestamp(),
      });
    }

    // Handle components
    const existingCompsSnap = await db.collection("logComponents").where("logItemId", "==", logItemId).get();
    const existingCompIds = new Set(existingCompsSnap.docs.map((doc) => doc.id));
    const incomingCompIds = new Set(logItem.components.filter((comp: any) => comp.id).map((comp: any) => comp.id));

    for (const compDoc of existingCompsSnap.docs) {
      if (!incomingCompIds.has(compDoc.id)) {
        await compDoc.ref.delete();
      }
    }

    if (Array.isArray(logItem.components)) {
      for (const comp of logItem.components) {
        let compRef;
        if (comp.id && existingCompIds.has(comp.id)) {
          compRef = db.collection("logComponents").doc(comp.id);
          await compRef.update({
            partNo: comp.partNo || "",
            serialOn: comp.serialOn || "",
            partOff: comp.partOff || "",
            serialOff: comp.serialOff || "",
            grn: comp.grn || "",
            updatedAt: admin.firestore.FieldValue.serverTimestamp(),
          });
        } else {
          compRef = db.collection("logComponents").doc();
          await compRef.set({
            logItemId,
            partNo: comp.partNo || "",
            serialOn: comp.serialOn || "",
            partOff: comp.partOff || "",
            serialOff: comp.serialOff || "",
            grn: comp.grn || "",
            createdAt: admin.firestore.FieldValue.serverTimestamp(),
            updatedAt: admin.firestore.FieldValue.serverTimestamp(),
          });
        }
      }
    }

    let savedDdNo = logItem.ddNo;
    if (logItem.ddChecked && logItem.ddAction) {
      const generatedDdNo = await DeferralService.handleDeferral(logItemId, logPageNo, logItem, createdBy);
      if (isNew && generatedDdNo && generatedDdNo !== logItem.ddNo) {
        await db.collection("logItems").doc(logItemId).update({ ddNo: generatedDdNo });
        savedDdNo = generatedDdNo;
      }
    }

    // Return direct object response
    return res.status(200).json({
      success: true,
      message: isNew ? "Log item added successfully ✅" : "Log item updated successfully ✅",
      data: { id: logItemId, displayNumber, ddNo: savedDdNo }
    });
  });

  saveLog = asyncHandler(async (req: Request, res: Response) => {
    const { logEntries, createdBy, logPageNo } = req.body;
    
    if (!logEntries || !Array.isArray(logEntries) || !logEntries.length) {
      throw new BadRequestError("Invalid logEntries data");
    }

    const savedLogs = [];
    const logRef = db.collection("logs").doc();
    const logId = logRef.id;
    const generatedLogPageNo = logPageNo || (await generateLogPageNo());

    await logRef.set({
      logPageNo: generatedLogPageNo,
      status: 1,
      createdBy: createdBy || "",
      createdAt: admin.firestore.FieldValue.serverTimestamp(),
      updatedAt: admin.firestore.FieldValue.serverTimestamp(),
    });

    const existingItemsSnap = await db.collection("logItems").where("logId", "==", logId).get();
    const maxDisplayNumber = existingItemsSnap.empty
      ? 0
      : Math.max(...existingItemsSnap.docs.map((doc) => doc.data().displayNumber || 0));

    for (const [index, entry] of logEntries.entries()) {
      const logItemRef = db.collection("logItems").doc();
      const logItemId = logItemRef.id;

      await logItemRef.set({
        logId,
        displayNumber: maxDisplayNumber + index + 1,
        raisedBy: entry.raisedBy || "",
        class: entry.class || "",
        defectDetails: entry.defectDetails || "",
        actionDetails: entry.actionDetails || "",
        ddChecked: entry.ddChecked || false,
        ddAction: entry.ddAction || "",
        ddType: normalizeDdType(entry.ddType) || "",
        ddNo: entry.ddNo || "",
        melCdlRef: entry.melCdlRef || "",
        cat: normalizeMelCat(entry.cat) || "A",
        indInspChecked: entry.indInspChecked || false,
        sdr: entry.sdr || false,
        mmsgFc: entry.mmsgFc || "",
        ata: entry.ata || "",
        createdAt: admin.firestore.FieldValue.serverTimestamp(),
        updatedAt: admin.firestore.FieldValue.serverTimestamp(),
      });

      if (Array.isArray(entry.components)) {
        for (const comp of entry.components) {
          const compRef = db.collection("logComponents").doc();
          await compRef.set({
            logItemId,
            partNo: comp.partNo || "",
            serialOn: comp.serialOn || "",
            partOff: comp.partOff || "",
            serialOff: comp.serialOff || "",
            grn: comp.grn || "",
            createdAt: admin.firestore.FieldValue.serverTimestamp(),
            updatedAt: admin.firestore.FieldValue.serverTimestamp(),
          });
        }
      }

      if (entry.ddChecked && entry.ddAction) {
        const generatedDdNo = await DeferralService.handleDeferral(logItemId, generatedLogPageNo, entry);
        if (generatedDdNo && generatedDdNo !== entry.ddNo) {
          await db.collection('logItems').doc(logItemId).update({ ddNo: generatedDdNo });
          entry.ddNo = generatedDdNo;
        }
      }

      savedLogs.push({ logItemId, displayNumber: maxDisplayNumber + index + 1, ddNo: entry.ddNo });
    }

    // Return direct object response
    return res.status(200).json({
      success: true,
      message: "Logs saved ✅",
      data: {
        logId,
        logPageNo: generatedLogPageNo,
        savedLogs,
      }
    });
  });

  getAll = asyncHandler(async (req: Request, res: Response) => {
    try {
      console.log("Fetching logs from Firestore...");

      // Fetch logs ordered by creation time (latest first)
      const logsSnapshot = await db
        .collection("logs")
        .orderBy("createdAt", "desc")
        .get();

      console.log("logsSnapshot empty:", logsSnapshot.empty);
      console.log("logsSnapshot size:", logsSnapshot.size);

      if (logsSnapshot.empty) {
        console.log("No logs found, returning empty array");
        return res.status(200).json([]);
      }

      const logs: any[] = [];

      // Iterate over logs
      for (const doc of logsSnapshot.docs) {
        const logData = doc.data();
        const logId = doc.id;
        console.log(`Processing log ${logId}:`, logData);

        // Fetch items belonging to this log
        const logItemsSnap = await db
          .collection("logItems")
          .where("logId", "==", logId)
          .get();

        console.log(`Found ${logItemsSnap.size} items for log ${logId}`);

        const logItems: any[] = [];

        // Iterate over items
        for (const itemDoc of logItemsSnap.docs) {
          const itemData = itemDoc.data();
          const logItemId = itemDoc.id;

          // Fetch components of this item
          const compsSnap = await db
            .collection("logComponents")
            .where("logItemId", "==", logItemId)
            .get();

          const components = compsSnap.docs.map((c) => ({
            id: c.id,
            ...c.data(),
          }));

          logItems.push({ id: logItemId, ...itemData, components });
        }

        // Sort logItems by displayNumber if available
        logItems.sort((a, b) => (a.displayNumber || 0) - (b.displayNumber || 0));

        logs.push({
          id: logId,
          ...logData,
          items: logItems,
        });
      }

      // Sort main logs array by displayNumber (override Firestore order)
      logs.sort((a, b) => (a.displayNumber || 0) - (b.displayNumber || 0));

      console.log("✅ Final logs (sorted by displayNumber):", logs.map(l => l.displayNumber));

      // Send JSON response
      return res.status(200).json(logs);
    } catch (error) {
      console.error("❌ Error in getAll:", error);
      return res.status(500).json({
        message: "Error fetching logs",
      });
    }
  });


  getOne = asyncHandler(async (req: Request, res: Response) => {
    const logId = req.params.id;

    const logDoc = await db.collection("logs").doc(logId).get();
    if (!logDoc.exists) {
      throw new BadRequestError("Log not found");
    }

    const logData = logDoc.data()!;

    const logItemsSnap = await db
      .collection("logItems")
      .where("logId", "==", logId)
      .get();
    const logItems = [];

    for (const itemDoc of logItemsSnap.docs) {
      const itemData = itemDoc.data();
      const logItemId = itemDoc.id;

      const compsSnap = await db
        .collection("logComponents")
        .where("logItemId", "==", logItemId)
        .get();
      const components = compsSnap.docs.map((c) => ({
        id: c.id,
        ...c.data(),
      }));

      logItems.push({ id: logItemId, ...itemData, components });
    }

    // Return direct object response
    return res.status(200).json({
      id: logId,
      ...logData,
      items: logItems,
    });
  });

  update = asyncHandler(async (req: Request, res: Response) => {
    const logId = req.params.id;
    const { logEntries, status, updatedBy } = req.body;

    if (!logEntries || !Array.isArray(logEntries)) {
      throw new BadRequestError('Invalid logEntries data');
    }

    const logRef = db.collection('logs').doc(logId);
    const logDoc = await logRef.get();
    if (!logDoc.exists) {
      throw new BadRequestError('Log not found');
    }

    const logPageNo = logDoc.data()!.logPageNo || (await generateLogPageNo());

    await logRef.update({
      status: status ?? logDoc.data()!.status,
      updatedBy: updatedBy || '',
      updatedAt: admin.firestore.FieldValue.serverTimestamp(),
    });

    const existingItemsSnap = await db.collection('logItems').where('logId', '==', logId).get();
    const existingItemIds = new Set(existingItemsSnap.docs.map((doc) => doc.id));
    const incomingItemIds = new Set(logEntries.filter((entry: any) => entry.id).map((entry: any) => entry.id));

    for (const itemDoc of existingItemsSnap.docs) {
      if (!incomingItemIds.has(itemDoc.id)) {
        const logItemId = itemDoc.id;
        const compsSnap = await db.collection('logComponents').where('logItemId', '==', logItemId).get();
        for (const compDoc of compsSnap.docs) {
          await compDoc.ref.delete();
        }
        await itemDoc.ref.delete();
      }
    }

    const newLogItems = [];
    for (const [index, entry] of logEntries.entries()) {
      let logItemId;
      let isNew = false;

      if (entry.id && existingItemIds.has(entry.id)) {
        logItemId = entry.id;
        const logItemRef = db.collection('logItems').doc(logItemId);
        await logItemRef.update({
          displayNumber: index + 1,
          raisedBy: entry.raisedBy || '',
          class: entry.class || '',
          defectDetails: entry.defectDetails || '',
          actionDetails: entry.actionDetails || '',
          ddChecked: entry.ddChecked || false,
          ddAction: entry.ddAction || '',
          ddType: normalizeDdType(entry.ddType) || '',
          ddNo: entry.ddNo || '',
          melCdlRef: entry.melCdlRef || '',
          cat: normalizeMelCat(entry.cat) || 'A',
          indInspChecked: entry.indInspChecked || false,
          sdr: entry.sdr || false,
          mmsgFc: entry.mmsgFc || '',
          ata: entry.ata || '',
          updatedAt: admin.firestore.FieldValue.serverTimestamp(),
        });
      } else {
        const logItemRef = db.collection('logItems').doc();
        logItemId = logItemRef.id;
        isNew = true;
        await logItemRef.set({
          logId,
          displayNumber: index + 1,
          raisedBy: entry.raisedBy || '',
          class: entry.class || '',
          defectDetails: entry.defectDetails || '',
          actionDetails: entry.actionDetails || '',
          ddChecked: entry.ddChecked || false,
          ddAction: entry.ddAction || '',
          ddType: normalizeDdType(entry.ddType) || '',
          ddNo: entry.ddNo || '',
          melCdlRef: entry.melCdlRef || '',
          cat: normalizeMelCat(entry.cat) || 'A',
          indInspChecked: entry.indInspChecked || false,
          sdr: entry.sdr || false,
          mmsgFc: entry.mmsgFc || '',
          ata: entry.ata || '',
          createdAt: admin.firestore.FieldValue.serverTimestamp(),
          updatedAt: admin.firestore.FieldValue.serverTimestamp(),
        });
      }

      if (entry.ddChecked && entry.ddAction) {
        const generatedDdNo = await DeferralService.handleDeferral(logItemId, logPageNo, entry);
        if (isNew && generatedDdNo && generatedDdNo !== entry.ddNo) {
          await db.collection('logItems').doc(logItemId).update({ ddNo: generatedDdNo });
          entry.ddNo = generatedDdNo;
        }
      }

      const existingCompsSnap = await db.collection('logComponents').where('logItemId', '==', logItemId).get();
      const existingCompIds = new Set(existingCompsSnap.docs.map((doc) => doc.id));
      const incomingCompIds = new Set((entry.components || []).filter((comp: any) => comp.id).map((comp: any) => comp.id));

      for (const compDoc of existingCompsSnap.docs) {
        if (!incomingCompIds.has(compDoc.id)) {
          await compDoc.ref.delete();
        }
      }

      for (const comp of entry.components || []) {
        let compRef;
        if (comp.id && existingCompIds.has(comp.id)) {
          compRef = db.collection('logComponents').doc(comp.id);
          await compRef.update({
            partNo: comp.partNo || '',
            serialOn: comp.serialOn || '',
            partOff: comp.partOff || '',
            serialOff: comp.serialOff || '',
            grn: comp.grn || '',
            updatedAt: admin.firestore.FieldValue.serverTimestamp(),
          });
        } else {
          compRef = db.collection('logComponents').doc();
          await compRef.set({
            logItemId,
            partNo: comp.partNo || '',
            serialOn: comp.serialOn || '',
            partOff: comp.partOff || '',
            serialOff: comp.serialOff || '',
            grn: comp.grn || '',
            createdAt: admin.firestore.FieldValue.serverTimestamp(),
            updatedAt: admin.firestore.FieldValue.serverTimestamp(),
          });
        }
      }

      newLogItems.push({ id: logItemId, displayNumber: index + 1, ddNo: entry.ddNo });
    }

    // Return direct object response
    return res.status(200).json({
      success: true,
      message: 'Log updated ✅',
      data: {
        logId,
        newLogItems,
      }
    });
  });

  delete = asyncHandler(async (req: Request, res: Response) => {
    const logId = req.params.id;

    const logRef = db.collection("logs").doc(logId);
    const logDoc = await logRef.get();
    if (!logDoc.exists) {
      throw new BadRequestError("Log not found");
    }

    const logItemsSnap = await db.collection("logItems").where("logId", "==", logId).get();
    for (const itemDoc of logItemsSnap.docs) {
      const logItemId = itemDoc.id;

      const compsSnap = await db.collection("logComponents").where("logItemId", "==", logItemId).get();
      for (const compDoc of compsSnap.docs) {
        await compDoc.ref.delete();
      }

      const deferralsSnap = await db
        .collection("deferrals")
        .where("entries.defect_reference.log_item_no", "==", logItemId)
        .get();
      for (const deferralDoc of deferralsSnap.docs) {
        await deferralDoc.ref.delete();
      }

      await itemDoc.ref.delete();
    }

    await logRef.delete();

    // Return direct object response
    return res.status(200).json({
      success: true,
      message: "Log deleted ✅",
      data: { logId }
    });
  });

  deleteLogItem = asyncHandler(async (req: Request, res: Response) => {
    const { logId, itemId } = req.params;

    const logRef = db.collection("logs").doc(logId);
    const logDoc = await logRef.get();
    if (!logDoc.exists) {
      throw new BadRequestError("Log not found");
    }

    const logItemRef = db.collection("logItems").doc(itemId);
    const logItemDoc = await logItemRef.get();
    if (!logItemDoc.exists) {
      throw new BadRequestError("Log item not found");
    }

    const compsSnap = await db.collection("logComponents").where("logItemId", "==", itemId).get();
    for (const compDoc of compsSnap.docs) {
      await compDoc.ref.delete();
    }

    const deferralsSnap = await db
      .collection("deferrals")
      .where("entries.defect_reference.log_item_no", "==", itemId)
      .get();
    for (const deferralDoc of deferralsSnap.docs) {
      await deferralDoc.ref.delete();
    }

    await logItemRef.delete();

    // Return direct object response
    return res.status(200).json({
      success: true,
      message: "Log item deleted successfully ✅",
      data: { logItemId: itemId }
    });
  });
}