import asyncHandler from "../../helpers/asyncHandler";
import { Request, Response, NextFunction } from "express";
import { SuccessResponse } from "../../utils/ApiResponse";
import { usersCollection, UserRole } from "../../model/user.model";
import PasswordUtils from "../../utils/passwordUtils";
import JwtUtils from "../../utils/jwtUtils";
import { BadRequestError } from "../../utils/ApiError";
import { admin } from "../../../firebase";

export default class UserController {
  // ✅ Get all users
  getAll = asyncHandler(async (req: any, res: Response, next: NextFunction) => {
    if (req.user.role !== UserRole.ADMIN) {
      throw new BadRequestError("Unauthorized");
    }

    const snapshot = await usersCollection.get();
    const users = snapshot.docs.map((doc) => ({ id: doc.id, ...doc.data() }));

    return new SuccessResponse("Users fetched successfully", users).send(res);
  });

  // ✅ Get one user
  getOne = asyncHandler(async (req: any, res: Response, next: NextFunction) => {
    const userId =
      req.user.role === UserRole.ADMIN ? req.params.id : req.user._id;

    const doc = await usersCollection.doc(userId).get();
    if (!doc.exists) throw new BadRequestError("User not found");

    return new SuccessResponse("User fetched successfully", {
      id: doc.id,
      ...doc.data(),
    }).send(res);
  });

  // ✅ Create a new user
  create = asyncHandler(async (req: Request, res: Response, next: NextFunction) => {
    const { phoneNumber, password } = req.body;

    if (!phoneNumber) {
      throw new BadRequestError("Phone number is required");
    }

    const existing = await usersCollection
      .where("phoneNumber", "==", phoneNumber)
      .where("deleted", "==", false)
      .get();

    if (!existing.empty) {
      throw new BadRequestError("User already exists");
    }

    const hashedPassword = password
      ? await PasswordUtils.hashPassword(password)
      : "";

    const now = new Date();
    const newUser = {
      ...req.body,
      password: hashedPassword,
      deleted: false,
      active: true,
      role: req.body.role || UserRole.USER,
      createdAt: now,
      updatedAt: now,
    };

    const docRef = await usersCollection.add(newUser);
    const token = JwtUtils.generateToken({ _id: docRef.id, role: newUser.role });

    return new SuccessResponse("User created successfully", {
      user: { id: docRef.id, ...newUser },
      token,
    }).send(res);
  });

  // ✅ Update user
  update = asyncHandler(async (req: any, res: Response, next: NextFunction) => {
    const userId =
      req.user.role === UserRole.ADMIN ? req.params.id : req.user._id;

    if (req.body.password) {
      req.body.password = await PasswordUtils.hashPassword(req.body.password);
    }

    await usersCollection.doc(userId).update({
      ...req.body,
      updatedAt: new Date(),
    });

    const updatedDoc = await usersCollection.doc(userId).get();

    return new SuccessResponse("User updated successfully", {
      id: updatedDoc.id,
      ...updatedDoc.data(),
    }).send(res);
  });

  // ✅ Delete user (soft delete)
  delete = asyncHandler(async (req: any, res: Response, next: NextFunction) => {
    const userId =
      req.user.role === UserRole.ADMIN ? req.params.id : req.user._id;

    await usersCollection.doc(userId).update({
      active: false,
      deleted: true,
      updatedAt: new Date(),
    });

    return new SuccessResponse("User deleted successfully", null).send(res);
  });

  // ✅ Sign In
  signIn = asyncHandler(async (req: Request, res: Response, next: NextFunction) => {
    const { phoneNumber, password } = req.body;

    const snapshot = await usersCollection
      .where("phoneNumber", "==", phoneNumber)
      .limit(1)
      .get();

    if (snapshot.empty) {
      throw new BadRequestError("User not found");
    }

    const doc = snapshot.docs[0];
    const user = doc.data();

    const isValid = await PasswordUtils.comparePassword(password, user.password);
    if (!isValid) throw new BadRequestError("Invalid credentials");

    const token = JwtUtils.generateToken({ _id: doc.id, role: user.role });

    return new SuccessResponse("User signed in successfully", {
      user: { id: doc.id, ...user },
      token,
    }).send(res);
  });

  // ✅ Forget password (OTP simulation)
  forgetPassword = asyncHandler(async (req: Request, res: Response, next: NextFunction) => {
    const snapshot = await usersCollection
      .where("phoneNumber", "==", req.body.phoneNumber)
      .limit(1)
      .get();

    if (snapshot.empty) {
      throw new BadRequestError("User not found");
    }

    const otp = Math.floor(100000 + Math.random() * 900000);
    // Normally send via SMS/email here
    return new SuccessResponse("OTP sent successfully", { otp }).send(res);
  });

  // ✅ Reset password (with fixed OTP for demo)
  resetPassword = asyncHandler(async (req: Request, res: Response, next: NextFunction) => {
    const snapshot = await usersCollection
      .where("phoneNumber", "==", req.body.phoneNumber)
      .limit(1)
      .get();

    if (snapshot.empty) {
      throw new BadRequestError("User not found");
    }

    const doc = snapshot.docs[0];
    const userId = doc.id;
    const systemOtp = 1000;

    if (systemOtp === req.body.otp) {
      const newPassword = await PasswordUtils.hashPassword(req.body.password);
      await usersCollection.doc(userId).update({
        password: newPassword,
        updatedAt: new Date(),
      });
      return new SuccessResponse("Password reset successfully", null).send(res);
    }

    throw new BadRequestError("Invalid OTP");
  });

  // ✅ Create Firebase Auth user for existing Firestore user (migration helper)
  createFirebaseAuthUser = asyncHandler(async (req: Request, res: Response, next: NextFunction) => {
    const { email, password } = req.body;

    if (!email || !password) {
      throw new BadRequestError("Email and password are required");
    }

    const normalizedEmail = email.toLowerCase().trim();

    // Check if user exists in Firestore
    const allUsersSnapshot = await usersCollection
      .where("deleted", "==", false)
      .get();
    
    const matchingUser = allUsersSnapshot.docs.find(doc => {
      const userData = doc.data();
      const userEmail = userData.email?.toLowerCase().trim();
      return userEmail === normalizedEmail;
    });

    if (!matchingUser) {
      throw new BadRequestError("User not found in system. Please sign up first.");
    }

    const userData = matchingUser.data();

    // Check if password matches (if stored)
    if (userData.password) {
      const isValid = await PasswordUtils.comparePassword(password, userData.password);
      if (!isValid) {
        throw new BadRequestError("Invalid password");
      }
    }

    // Try to create Firebase Auth user
    try {
      // Note: We can't create Firebase Auth users from backend directly
      // This is a limitation - users must be created from client
      // Instead, we'll return a token that allows the client to create the user
      // Or we can check if user already has firebaseUid
      
      if (userData.firebaseUid) {
        // User already has Firebase Auth account
        return new SuccessResponse("User already has Firebase Auth account", {
          email: normalizedEmail,
          firebaseUid: userData.firebaseUid,
          message: "Please use Firebase Auth login"
        }).send(res);
      }

      // Return success with instructions
      return new SuccessResponse("User found. Please create Firebase Auth account on mobile app.", {
        email: normalizedEmail,
        userId: matchingUser.id,
        message: "User exists in system. Use 'Sign Up' on mobile app with same email/password."
      }).send(res);
    } catch (error: any) {
      throw new BadRequestError(`Failed to process: ${error?.message}`);
    }
  });

  // ✅ Exchange Firebase token for JWT token
  exchangeFirebaseToken = asyncHandler(async (req: Request, res: Response, next: NextFunction) => {
    const { firebaseToken } = req.body;

    if (!firebaseToken) {
      throw new BadRequestError("Firebase token is required");
    }

    try {
      // Verify Firebase token
      const decodedToken = await admin.auth().verifyIdToken(firebaseToken);
      const firebaseUid = decodedToken.uid;
      const email = decodedToken.email?.toLowerCase().trim(); // Normalize email
      const displayName = decodedToken.name || decodedToken.display_name || "";

      // Try to find user by Firebase UID or email
      let userDoc = null;
      let userId = null;

      // First, try to find by Firebase UID
      const byUidSnapshot = await usersCollection
        .where("firebaseUid", "==", firebaseUid)
        .where("deleted", "==", false)
        .limit(1)
        .get();

      if (!byUidSnapshot.empty) {
        userDoc = byUidSnapshot.docs[0];
        userId = userDoc.id;
      } else if (email) {
        // Try to find by email (case-insensitive)
        // First try exact match (most common case)
        const byEmailSnapshot = await usersCollection
          .where("email", "==", email)
          .where("deleted", "==", false)
          .limit(1)
          .get();

        if (!byEmailSnapshot.empty) {
          userDoc = byEmailSnapshot.docs[0];
          userId = userDoc.id;
          // Update user with Firebase UID and ensure email is normalized
          await usersCollection.doc(userId).update({
            firebaseUid,
            email: email, // Store normalized email
            updatedAt: new Date(),
          });
        } else {
          // Fallback: try case-insensitive search (only if exact match fails)
          // This is less efficient but handles edge cases
          const allUsersSnapshot = await usersCollection
            .where("deleted", "==", false)
            .get();
          
          const matchingUser = allUsersSnapshot.docs.find(doc => {
            const userData = doc.data();
            const userEmail = userData.email?.toLowerCase().trim();
            return userEmail === email;
          });

          if (matchingUser) {
            userDoc = matchingUser;
            userId = matchingUser.id;
            // Update user with Firebase UID and ensure email is normalized
            await usersCollection.doc(userId).update({
              firebaseUid,
              email: email, // Store normalized email
              updatedAt: new Date(),
            });
          }
        }
      }

      // If user doesn't exist, create a new one
      if (!userDoc) {
        const now = new Date();
        const newUser = {
          firebaseUid,
          email: email || "",
          name: displayName || "",
          fullName: displayName || "", // Also store as fullName for compatibility
          role: UserRole.USER,
          active: true,
          deleted: false,
          createdAt: now,
          updatedAt: now,
        };

        const docRef = await usersCollection.add(newUser);
        userId = docRef.id;
        userDoc = await usersCollection.doc(userId).get();
      } else {
        // Update existing user with latest info from Firebase
        const userData = userDoc.data();
        const updates: any = {
          updatedAt: new Date(),
        };

        // Update name if Firebase has a display name and we don't have one
        if (displayName && !userData.name && !userData.fullName) {
          updates.name = displayName;
          updates.fullName = displayName;
        }

        // Normalize email if needed
        if (email && userData.email?.toLowerCase().trim() !== email) {
          updates.email = email;
        }

        // Ensure firebaseUid is set
        if (!userData.firebaseUid) {
          updates.firebaseUid = firebaseUid;
        }

        if (Object.keys(updates).length > 1) { // More than just updatedAt
          await usersCollection.doc(userId).update(updates);
          userDoc = await usersCollection.doc(userId).get(); // Refresh
        }
      }

      const userData = userDoc.data();
      
      // Generate JWT token
      const jwtToken = JwtUtils.generateToken({ 
        _id: userId, 
        role: userData?.role || UserRole.USER 
      });

      // Return response with token and user data
      return new SuccessResponse("Token exchanged successfully", {
        token: jwtToken,
        user: { id: userId, ...userData },
      }).send(res);
    } catch (error: any) {
      console.error("Exchange Firebase Token Error:", error);
      throw new BadRequestError(`Invalid Firebase token: ${error?.message}`);
    }
  });
}
