export declare abstract class Expression<T extends string | number | boolean | string[]> {
    value(): T;
    toCEL(): string;
    toJSON(): string;
}
/**
 * A CEL expression corresponding to a ternary operator, e.g {{ cond ? ifTrue : ifFalse }}
 */
export declare class TernaryExpression<T extends string | number | boolean | string[]> extends Expression<T> {
    private readonly test;
    private readonly ifTrue;
    private readonly ifFalse;
    constructor(test: Expression<boolean>, ifTrue: T, ifFalse: T);
    value(): T;
    toString(): string;
}
/**
 * A CEL expression that evaluates to boolean true or false based on a comparison
 * between the value of another expression and a literal of that same type.
 */
export declare class CompareExpression<T extends string | number | boolean | string[]> extends Expression<boolean> {
    cmp: '==' | '>' | '>=' | '<' | '<=';
    lhs: Expression<T>;
    rhs: T;
    constructor(cmp: '==' | '>' | '>=' | '<' | '<=', lhs: Expression<T>, rhs: T);
    value(): boolean;
    toString(): string;
    then(ifTrue: T, ifFalse: T): TernaryExpression<T>;
}
/** @hidden */
declare type ParamValueType = 'string' | 'list' | 'boolean' | 'int' | 'float' | 'secret';
declare type ParamInput<T> = {
    text: TextInput<T>;
} | {
    select: SelectInput<T>;
} | {
    resource: ResourceInput;
};
/**
 * Specifies that a Param's value should be determined by prompting the user
 * to type it in interactively at deploy-time. Input that does not match the
 * provided validationRegex, if present, will be retried.
 */
export interface TextInput<T = unknown> {
    example?: string;
    validationRegex?: string;
    validationErrorMessage?: string;
}
/**
 * Specifies that a Param's value should be determined by having the user
 * select from a list containing all the project's resources of a certain
 * type. Currently, only type:"storage.googleapis.com/Bucket" is supported.
 */
export interface ResourceInput {
    resource: {
        type: string;
    };
}
/**
 * Specifies that a Param's value should be determined by having the user select
 * from a list of pre-canned options interactively at deploy-time.
 */
export interface SelectInput<T = unknown> {
    options: Array<SelectOptions<T>>;
}
export interface SelectOptions<T = unknown> {
    label?: string;
    value: T;
}
export interface ParamSpec<T = unknown> {
    name: string;
    default?: T;
    label?: string;
    description?: string;
    type: ParamValueType;
    input?: ParamInput<T>;
}
export declare type ParamOptions<T = unknown> = Omit<ParamSpec<T>, 'name' | 'type'>;
export declare abstract class Param<T extends string | number | boolean | string[]> extends Expression<T> {
    readonly name: string;
    readonly options: ParamOptions<T>;
    static type: ParamValueType;
    constructor(name: string, options?: ParamOptions<T>);
    value(): T;
    cmp(cmp: '==' | '>' | '>=' | '<' | '<=', rhs: T): CompareExpression<T>;
    equals(rhs: T): CompareExpression<T>;
    toString(): string;
    toSpec(): ParamSpec<T>;
}
export declare class SecretParam {
    static type: ParamValueType;
    name: string;
    constructor(name: string);
    value(): string;
    toSpec(): ParamSpec<string>;
}
export declare class StringParam extends Param<string> {
    value(): string;
}
export declare class IntParam extends Param<number> {
    static type: ParamValueType;
    value(): number;
}
export declare class FloatParam extends Param<number> {
    static type: ParamValueType;
    value(): number;
}
export declare class BooleanParam extends Param<boolean> {
    static type: ParamValueType;
    value(): boolean;
    then<T extends string | number | boolean>(ifTrue: T, ifFalse: T): TernaryExpression<T>;
}
export declare class ListParam extends Param<string[]> {
    static type: ParamValueType;
    value(): string[];
    toSpec(): ParamSpec<string[]>;
}
export {};
