import asyncHandler from "../../helpers/asyncHandler";
import { Request, Response } from "express";
import { BadRequestError } from "../../utils/ApiError";
import { db, admin } from "../../../firebase"; // Removed .ts extension
import {
  S3Client,
  PutObjectCommand,
  DeleteObjectCommand,
  GetObjectCommand,
} from "@aws-sdk/client-s3";
import { getSignedUrl } from "@aws-sdk/s3-request-presigner";

// Make S3 optional; if env is missing, skip S3 operations gracefully
const AWS_REGION = process.env.AWS_REGION;
const AWS_ACCESS_KEY_ID = process.env.AWS_ACCESS_KEY_ID;
const AWS_SECRET_ACCESS_KEY = process.env.AWS_SECRET_ACCESS_KEY;
const AWS_BUCKET_NAME = process.env.AWS_BUCKET_NAME;
const hasS3 = Boolean(
  AWS_REGION && AWS_ACCESS_KEY_ID && AWS_SECRET_ACCESS_KEY && AWS_BUCKET_NAME
);

const s3 = hasS3
  ? new S3Client({
      region: AWS_REGION,
      credentials: {
        accessKeyId: AWS_ACCESS_KEY_ID as string,
        secretAccessKey: AWS_SECRET_ACCESS_KEY as string,
      },
      forcePathStyle: true, // ← Add this line
    })
  : null;

export default class DocumentController {
  
  create = asyncHandler(async (req: any, res: Response) => {
    const { docName, docType, description } = req.body;
    const file = req.file;

    if (!file) {
      throw new BadRequestError("No file uploaded");
    }

    if (!docName || !docType) {
      throw new BadRequestError("Document name and type are required");
    }

    const safeName = file.originalname.replace(/\s+/g, "_");
    const uniqueName = Date.now() + "-" + safeName;

    if (!hasS3 || !s3) {
      throw new BadRequestError(
        "S3 is not configured. Please set AWS_REGION, AWS_ACCESS_KEY_ID, AWS_SECRET_ACCESS_KEY, AWS_BUCKET_NAME"
      );
    }

    await s3.send(
      new PutObjectCommand({
        Bucket: AWS_BUCKET_NAME!,
        Key: uniqueName,
        Body: file.buffer,
        ContentType: file.mimetype,
      })
    );

    const fileKey = uniqueName;

    const docRef = await db.collection("documents").add({
      docName,
      docType,
      description,
      fileKey,
      fileName: file.originalname,
      fileSize: file.size,
      createdAt: admin.firestore.FieldValue.serverTimestamp(),
    });

    // Return direct object response
    return res.status(200).json({
      success: true,
      message: "File uploaded to S3 (private) and metadata saved ✅", 
      data: { docId: docRef.id, fileKey }
    });
  });

  getAll = asyncHandler(async (req: Request, res: Response) => {
    const snapshot = await db
      .collection("documents")
      .orderBy("createdAt", "desc")
      .get();

    const docs = await Promise.all(
      snapshot.docs.map(async (doc) => {
        const data = doc.data();
        let signedUrl = null;

        if (data.fileKey && hasS3 && s3) {
          const command = new GetObjectCommand({
            Bucket: AWS_BUCKET_NAME!,
            Key: data.fileKey,
          });
          signedUrl = await getSignedUrl(s3, command, { expiresIn: 3600 });
        } else if (data.fileUrl) {
          signedUrl = data.fileUrl;
        }

        return {
          id: doc.id,
          ...data,
          fileUrl: signedUrl,
        };
      })
    );

    // Return direct array response
    // Return consistent format
    return res.status(200).json({
      success: true,
      data: docs
    });
  });

  delete = asyncHandler(async (req: Request, res: Response) => {
    const { id } = req.params;
    const docRef = db.collection("documents").doc(id);
    const doc = await docRef.get();

    if (!doc.exists) {
      throw new BadRequestError("Document not found");
    }

    const { fileKey, fileUrl } = doc.data()!;

    if (fileKey && hasS3 && s3) {
      await s3.send(
        new DeleteObjectCommand({
          Bucket: AWS_BUCKET_NAME!,
          Key: fileKey,
        })
      );
    } else if (fileUrl) {
      const key = decodeURIComponent(fileUrl.split("/").pop()!);
      if (hasS3 && s3) {
        await s3.send(
          new DeleteObjectCommand({
            Bucket: AWS_BUCKET_NAME!,
            Key: key,
          })
        );
      }
    }

    await docRef.delete();

    // Return direct object response
    return res.status(200).json({
      success: true,
      message: "Document and file deleted successfully ✅"
    });
  });

  // Optional: Add getOne method for single document
  getOne = asyncHandler(async (req: Request, res: Response) => {
    const { id } = req.params;
    const docRef = db.collection("documents").doc(id);
    const doc = await docRef.get();

    if (!doc.exists) {
      throw new BadRequestError("Document not found");
    }

    const data = doc.data()!;
    let signedUrl = null;

    if (data.fileKey) {
      const command = new GetObjectCommand({
        Bucket: process.env.AWS_BUCKET_NAME!,
        Key: data.fileKey,
      });
      signedUrl = await getSignedUrl(s3, command, { expiresIn: 3600 });
    } else if (data.fileUrl) {
      signedUrl = data.fileUrl;
    }

    // Return direct object response
    return res.status(200).json({
      id: doc.id,
      ...data,
      fileUrl: signedUrl,
    });
  });

  // Optional: Add update method for document metadata
  update = asyncHandler(async (req: Request, res: Response) => {
    const { id } = req.params;
    const { docName, docType, description } = req.body;

    const docRef = db.collection("documents").doc(id);
    const doc = await docRef.get();

    if (!doc.exists) {
      throw new BadRequestError("Document not found");
    }

    await docRef.update({
      docName,
      docType,
      description,
      updatedAt: admin.firestore.FieldValue.serverTimestamp(),
    });

    // Return direct object response
    return res.status(200).json({
      success: true,
      message: "Document updated successfully ✅"
    });
  });
}