import { admin } from "../../firebase";  // adjust path if needed


export enum UserRole {
  USER = "user",
  ADMIN = "admin",
}

// Initialize Firestore
const db = admin.firestore();
export const usersCollection = db.collection("users");

export interface User {
  id?: string;
  fullName: string;
  profilePicUrl?: string;
  email?: string;
  password: string;
  role?: UserRole;
  district?: string;
  dateOfBirth?: Date;
  phoneNumber: string;
  meta?: Record<string, any>;
  deleted?: boolean;
  active?: boolean;
  createdAt?: FirebaseFirestore.Timestamp;
  updatedAt?: FirebaseFirestore.Timestamp;
}

// ✅ Create a new user
export const createUser = async (data: User) => {
  const now = admin.firestore.Timestamp.now();
  const userData = {
    ...data,
    role: data.role || UserRole.USER,
    deleted: data.deleted ?? false,
    active: data.active ?? true,
    createdAt: now,
    updatedAt: now,
  };
  const docRef = await usersCollection.add(userData);
  return { id: docRef.id, ...userData };
};

// ✅ Get a user by ID
export const getUserById = async (id: string) => {
  const doc = await usersCollection.doc(id).get();
  if (!doc.exists) return null;
  return { id: doc.id, ...doc.data() } as User;
};

// ✅ Update a user
export const updateUser = async (id: string, updates: Partial<User>) => {
  await usersCollection.doc(id).update({
    ...updates,
    updatedAt: admin.firestore.Timestamp.now(),
  });
  return getUserById(id);
};

// ✅ Delete a user
export const deleteUser = async (id: string) => {
  await usersCollection.doc(id).delete();
  return true;
};
