import { db, admin } from "../../firebase";
import { normalizeDdType, normalizeMelCat, generateDeferralTypeNo } from "../utils/logUtils";

export class DeferralService {
  
  static async handleDeferral(logItemId: string, logPageNo: string, entry: any, createdBy: string = '') {
    let ddNo = entry.ddNo;

    const existingDeferralSnap = await db
      .collection('deferrals')
      .where('ddNo', '==', ddNo)
      .get();

    if (entry.ddAction === 'Raised (R)') {
      if (existingDeferralSnap.empty) {
        if (!ddNo) {
          ddNo = await generateDeferralTypeNo();
        }

        const existingDdNoSnap = await db
          .collection('deferrals')
          .where('entries.defect_reference.type_no', '==', ddNo)
          .get();

        if (!existingDdNoSnap.empty) {
          return null;
        }

        const deferralRef = db.collection('deferrals').doc();
        const deferralData = {
          ddNo,
          entries: [
            {
              defect_reference: {
                date: new Date().toISOString().split('T')[0],
                dd_action: entry.ddAction || '',
                dd: normalizeDdType(entry.ddType) || 'N',
                log_item_no: logItemId,
                log_page: logPageNo,
                mel_cat: normalizeMelCat(entry.cat) || 'A',
                mel_cd_ref: entry.melCdlRef || '',
                type_no: ddNo,
              },
              description: entry.defectDetails || 'No description provided',
              clear_reference: {},
              enteredSign: '',
              enteredAuth: '',
              enteredAuthName: '',
              enteredDate: '',
              expDate: '',
              clearedSign: '',
              clearedAuth: '',
              clearedAuthName: '',
              clearedDate: '',
              deferral: true,
            },
          ],
          createdBy: createdBy || '',
          createdAt: admin.firestore.FieldValue.serverTimestamp(),
          updatedAt: admin.firestore.FieldValue.serverTimestamp(),
        };

        await deferralRef.set(deferralData);

        await db.collection('deferralHistory').doc().set({ 
          deferralId: deferralRef.id,
          logItemId,
          ddNo,
          action: 'Created',
          description: entry.defectDetails || 'No description provided',
          data: deferralData,
          createdBy: createdBy || '',
          createdAt: admin.firestore.FieldValue.serverTimestamp(),
        });

        return ddNo;
      } else {
        const deferralDoc = existingDeferralSnap.docs[0];
        const deferralData = deferralDoc.data();

        await db.collection('deferralHistory').doc().set({
          deferralId: deferralDoc.id,
          logItemId,
          ddNo,
          action: 'Updated',
          description: entry.defectDetails || deferralData.entries[0].description || 'No description provided',
          data: deferralData,
          createdBy: createdBy || '',
          createdAt: admin.firestore.FieldValue.serverTimestamp(),
        });

        await deferralDoc.ref.update({
          'entries.0.defect_reference.dd_action': entry.ddAction || '',
          'entries.0.defect_reference.dd': normalizeDdType(entry.ddType) || 'N',
          'entries.0.defect_reference.mel_cat': normalizeMelCat(entry.cat) || 'A',
          'entries.0.defect_reference.mel_cd_ref': entry.melCdlRef || '',
          'entries.0.defect_reference.log_item_no': logItemId,
          'entries.0.defect_reference.log_page': logPageNo,
          'entries.0.description': entry.defectDetails || deferralData.entries[0].description || 'No description provided',
          updatedBy: createdBy || '',
          updatedAt: admin.firestore.FieldValue.serverTimestamp(),
        });

        return ddNo;
      }
    } else if (entry.ddAction === 'Cleared (C)' && ddNo) {
      if (!existingDeferralSnap.empty) {
        const deferralDoc = existingDeferralSnap.docs[0];
        const deferralData = deferralDoc.data();
        console.log("entry services:", entry)
        // Prepare clearance reference data
        const clearReference = {
          staff_id: entry.shortSignAuthId || '',
          staff_name: entry.shortSignAuthName || '',
          date: entry.clearedDate || new Date().toISOString().split('T')[0],
          log_page: logPageNo || '',
          log_item_no: logItemId || '',
        };

        await db.collection('deferralHistory').doc().set({
          deferralId: deferralDoc.id,
          logItemId,
          ddNo,
          action: 'Cleared',
          description: 'Deferral cleared with reference',
          data: deferralData,
          createdBy: createdBy || '',
          createdAt: admin.firestore.FieldValue.serverTimestamp(),
        });

        // Update with clearance data
        await deferralDoc.ref.update({
          'entries.0.defect_reference.dd_action': entry.ddAction || '',
          'entries.0.defect_reference.dd': normalizeDdType(entry.ddType) || 'N',
          'entries.0.defect_reference.mel_cat': normalizeMelCat(entry.cat) || 'A',
          'entries.0.defect_reference.mel_cd_ref': entry.melCdlRef || '',
          'entries.0.defect_reference.log_item_no': logItemId,
          'entries.0.defect_reference.log_page': logPageNo,
          'entries.0.description': entry.defectDetails || deferralData.entries[0].description || 'No description provided',
          'entries.0.clear_reference': clearReference,
          'entries.0.clearedSign': entry.clearedSign || '',
          'entries.0.clearedAuth': entry.clearedAuth || '',
          'entries.0.clearedAuthName': entry.clearedAuthName || '',
          'entries.0.clearedDate': entry.clearedDate || '',
          updatedBy: createdBy || '',
          updatedAt: admin.firestore.FieldValue.serverTimestamp(),
        });

        return ddNo;
      } else {
        return null;
      }
    } else if (entry.ddAction === 'Worked (W)' && ddNo) {
      if (!existingDeferralSnap.empty) {
        const deferralDoc = existingDeferralSnap.docs[0];
        const deferralData = deferralDoc.data();

        await db.collection('deferralHistory').doc().set({
          deferralId: deferralDoc.id,
          logItemId,
          ddNo,
          action: 'Updated',
          description: entry.defectDetails || deferralData.entries[0].description || 'No description provided',
          data: deferralData,
          createdBy: createdBy || '',
          createdAt: admin.firestore.FieldValue.serverTimestamp(),
        });

        await deferralDoc.ref.update({
          'entries.0.defect_reference.dd_action': entry.ddAction || '',
          'entries.0.defect_reference.dd': normalizeDdType(entry.ddType) || 'N',
          'entries.0.defect_reference.mel_cat': normalizeMelCat(entry.cat) || 'A',
          'entries.0.defect_reference.mel_cd_ref': entry.melCdlRef || '',
          'entries.0.defect_reference.log_item_no': logItemId,
          'entries.0.defect_reference.log_page': logPageNo,
          'entries.0.description': entry.defectDetails || deferralData.entries[0].description || 'No description provided',
          updatedBy: createdBy || '',
          updatedAt: admin.firestore.FieldValue.serverTimestamp(),
        });

        return ddNo;
      } else {
        return null;
      }
    } else {
      return null;
    }
  }
}